<?php

namespace App\Http\Controllers;

use App\Models\Gaji;
use App\Models\Pegawai;
use App\Models\UangMakan;
use Illuminate\Http\Request;
use Barryvdh\DomPDF\Facade\Pdf;
use App\Exports\GajiExport;
use Maatwebsite\Excel\Facades\Excel;
use Carbon\Carbon;

class GajiController extends Controller
{
    private function hitungPph21($penghasilanSetahun, $status_ptkp, $gaji_pokok)
    {
        $ptkp_map = [
    'TK/0' => 54000000,
    'TK/1' => 58500000,
    'TK/2' => 63000000,
    'TK/3' => 67500000,
    'K/0'  => 58500000,
    'K/1'  => 63000000,
    'K/2'  => 67500000,
    'K/3'  => 72000000,
        ];

        $ptkp = $ptkp_map[$status_ptkp] ?? 54000000;

        // Biaya Jabatan: 5% dari penghasilan, maksimal 6 juta
        $biaya_jabatan = min($penghasilanSetahun * 0.05, 6000000);

        // Iuran pensiun: 1% dari gaji pokok per bulan × 12
        $iuran_pensiun = $gaji_pokok * 0.01 * 12;

        // Penghasilan neto = bruto - biaya jabatan - iuran pensiun
        $neto = $penghasilanSetahun - $biaya_jabatan - $iuran_pensiun;

        // Penghasilan Kena Pajak
        $pkp = max($neto - $ptkp, 0);
        $pkp_bulat = floor($pkp / 1000) * 1000;

        // Hitung PPh 21 progresif
        $pph = 0;
        $lapisan = [
            [50000000, 0.05],
            [200000000, 0.15],
            [250000000, 0.25],
            [PHP_INT_MAX, 0.30],
        ];

        foreach ($lapisan as [$batas, $tarif]) {
            if ($pkp_bulat <= 0) break;
            $kena = min($pkp_bulat, $batas);
            $pph += $kena * $tarif;
            $pkp_bulat -= $kena;
        }

        // Hasil PPh 21 bulanan
        return round($pph / 12);
    }

    private function validateInput(Request $request)
    {
        $request->validate([
            'pegawai_id' => 'required|exists:pegawais,id',
            'bulan' => 'required|date_format:Y-m',
            'jumlah_hadir' => 'required|integer|min:0',
        ]);

        $maxHari = Carbon::parse($request->bulan . '-01')->endOfMonth()->day;
        if ($request->jumlah_hadir > $maxHari) {
            abort(400, 'Jumlah hadir tidak boleh lebih dari ' . $maxHari . ' hari.');
        }
    }

    private function simpanAtauUpdateGaji(Request $request, Gaji $gaji = null)
    {
        $pegawai = Pegawai::with('jabatan')->findOrFail($request->pegawai_id);
        $jumlah_hadir = $request->jumlah_hadir;

        $total_uang_makan = UangMakan::where('pegawai_id', $pegawai->id)
            ->where('bulan', $request->bulan)
            ->sum('jumlah');

        if ($total_uang_makan === 0) {
            session()->flash('warning', '⚠️ Data uang makan tidak ditemukan. Dianggap Rp 0.');
        }

        $gaji_pokok = $pegawai->jabatan->gaji_pokok ?? 0;
        $total_tunjangan = collect($request->tunjangan_jumlah)->sum();
        $potongan_tambahan = collect($request->potongan_jumlah)->sum();

        $bpjs_kesehatan = $gaji_pokok * 0.01;
        $bpjs_jht = $gaji_pokok * 0.02;
        $bpjs_jpens = $gaji_pokok * 0.01;

$pot_pph21_gaji = $this->hitungPph21(
    ($gaji_pokok + $total_uang_makan + $total_tunjangan) * 12,
    $pegawai->status_ptkp,
    $gaji_pokok
);
        $pot_pph21_non_gaji = 0;
        $pot_absen = $request->pot_absen ?? 0;
        $pot_pinjaman = $request->pot_pinjaman ?? 0;

        $jumlah_pendapatan = $gaji_pokok + $total_uang_makan + $total_tunjangan;
        $total_potongan = $bpjs_kesehatan + $bpjs_jht + $bpjs_jpens + $pot_pph21_gaji + $pot_pph21_non_gaji + $pot_absen + $pot_pinjaman + $potongan_tambahan;
        $total_gaji = $jumlah_pendapatan - $total_potongan;

        $data = [
            'pegawai_id' => $request->pegawai_id,
            'bulan' => $request->bulan,
            'jumlah_hadir' => $jumlah_hadir,
            'gaji_pokok' => $gaji_pokok,
            'total_uang_makan' => $total_uang_makan,
            'total_potongan' => $total_potongan,
            'total_gaji' => $total_gaji,
            'pot_pph21' => $pot_pph21_gaji,
            'pot_pph21_non_gaji' => $pot_pph21_non_gaji,
            'pot_absen' => $pot_absen,
            'pot_pinjaman' => $pot_pinjaman,
            'pot_bpjs_kesehatan' => $bpjs_kesehatan,
            'pot_bpjs_jht' => $bpjs_jht,
            'pot_bpjs_jpens' => $bpjs_jpens,
            'jumlah_pendapatan' => $jumlah_pendapatan,
        ];

        if ($gaji) {
            $gaji->update($data);
            $gaji->tunjangan()->delete();
            $gaji->potongan()->delete();
        } else {
            $gaji = Gaji::create($data);
        }

        foreach ($request->tunjangan_nama ?? [] as $i => $nama) {
            if ($nama && $request->tunjangan_jumlah[$i]) {
                $gaji->tunjangan()->create([
                    'nama' => $nama,
                    'jumlah' => $request->tunjangan_jumlah[$i],
                ]);
            }
        }

        foreach ($request->potongan_nama ?? [] as $i => $nama) {
            if ($nama && $request->potongan_jumlah[$i]) {
                $gaji->potongan()->create([
                    'nama' => $nama,
                    'jumlah' => $request->potongan_jumlah[$i],
                ]);
            }
        }

        return $gaji;
    }


    public function hitungUlangPph(Gaji $gaji)
{
    $pegawai = Pegawai::with('jabatan')->find($gaji->pegawai_id);
    if (!$pegawai || !$pegawai->jabatan) return;

    $jumlah_hadir = $gaji->jumlah_hadir;
    $total_uang_makan = UangMakan::where('pegawai_id', $pegawai->id)
        ->where('bulan', $gaji->bulan)
        ->sum('jumlah');
    $gaji_pokok = $pegawai->jabatan->gaji_pokok;
    $total_tunjangan = $gaji->tunjangan()->sum('jumlah');
    $potongan_tambahan = $gaji->potongan()->sum('jumlah');

    $penghasilan_setahun = ($gaji_pokok + $total_uang_makan + $total_tunjangan) * 12;
    $pot_pph21 = $this->hitungPph21($penghasilan_setahun, $pegawai->status_ptkp);

    $bpjs_kesehatan = $gaji_pokok * 0.01;
    $bpjs_jht = $gaji_pokok * 0.02;
    $bpjs_jpens = $gaji_pokok * 0.01;

    $pot_absen = $gaji->pot_absen ?? 0;
    $pot_pinjaman = $gaji->pot_pinjaman ?? 0;

    $jumlah_pendapatan = $gaji_pokok + $total_uang_makan + $total_tunjangan;
    $total_potongan = $bpjs_kesehatan + $bpjs_jht + $bpjs_jpens + $pot_pph21 + $pot_absen + $pot_pinjaman + $potongan_tambahan;
    $total_gaji = $jumlah_pendapatan - $total_potongan;

    $gaji->update([
        'gaji_pokok' => $gaji_pokok,
        'total_uang_makan' => $total_uang_makan,
        'jumlah_pendapatan' => $jumlah_pendapatan,
        'total_potongan' => $total_potongan,
        'total_gaji' => $total_gaji,
        'pot_pph21' => $pot_pph21,
        'pot_bpjs_kesehatan' => $bpjs_kesehatan,
        'pot_bpjs_jht' => $bpjs_jht,
        'pot_bpjs_jpens' => $bpjs_jpens,
    ]);
}

    public function store(Request $request)
    {
        $this->validateInput($request);
        $this->simpanAtauUpdateGaji($request);
        return redirect()->route('gaji.index')->with('success', 'Gaji berhasil ditambahkan.');
    }

    public function update(Request $request, Gaji $gaji)
    {
        $this->validateInput($request);
        $this->simpanAtauUpdateGaji($request, $gaji);
        return redirect()->route('gaji.index')->with('success', 'Data gaji berhasil diperbarui.');
    }

    public function show(Gaji $gaji)
    {
        $gaji->load(['pegawai', 'tunjangan', 'potongan']);
        return view('gaji.show', compact('gaji'));
    }

    public function create()
    {
        $pegawais = Pegawai::with('jabatan')->get();
        return view('gaji.create', compact('pegawais'));
    }

    public function edit(Gaji $gaji)
    {
        $pegawais = Pegawai::with('jabatan')->get();
        $gaji->load(['tunjangan', 'potongan']);
        return view('gaji.edit', compact('gaji', 'pegawais'));
    }

    public function destroy(Gaji $gaji)
    {
        $gaji->delete();
        return redirect()->route('gaji.index')->with('success', 'Data gaji berhasil dihapus.');
    }

    public function index(Request $request)
    {
        $periode = $request->query('periode') ?? now()->format('Y-m');
        $gajis = Gaji::with('pegawai')->where('bulan', $periode)->get();
        $semuaPeriode = Gaji::select('bulan')->distinct()->orderBy('bulan', 'desc')->pluck('bulan');
        return view('gaji.index', compact('gajis', 'periode', 'semuaPeriode'));
    }

public function slip(Gaji $gaji)
{
    // Pastikan relasi dimuat sebagai koleksi
    $gaji->loadMissing(['pegawai', 'tunjangan', 'potongan']);

    // Hindari kesalahan jika relasi belum ada
    $gaji->setRelation('tunjangan', $gaji->tunjangan ?? collect());
    $gaji->setRelation('potongan', $gaji->potongan ?? collect());

    $pdf = Pdf::loadView('gaji.slip', compact('gaji'));
    return $pdf->download('Slip-Gaji-' . $gaji->pegawai->nama . '.pdf');
}

    public function slipTotal(Pegawai $pegawai)
    {
        $gajis = Gaji::with(['tunjangan', 'potongan'])
            ->where('pegawai_id', $pegawai->id)
            ->orderBy('bulan')
            ->get();

        $totalGaji = $gajis->sum('gaji_pokok');
        $totalUangMakan = $gajis->sum('total_uang_makan');
        $totalTunjangan = $gajis->flatMap->tunjangan->sum('jumlah');
        $totalPotongan = $gajis->sum('total_potongan');
        $totalPotonganPPH = $gajis->sum('pot_pph21');

        $penghasilanKotor = $totalGaji + $totalUangMakan + $totalTunjangan;
        $biayaJabatan = min($penghasilanKotor * 0.05, 6000000);
        $penghasilanNeto = $penghasilanKotor - $biayaJabatan;

        $ptkpMap = [
    'TK/0' => 54000000,
    'TK/1' => 58500000,
    'TK/2' => 63000000,
    'TK/3' => 67500000,
    'K/0'  => 58500000,
    'K/1'  => 63000000,
    'K/2'  => 67500000,
    'K/3'  => 72000000,
        ];
        $ptkp = $ptkpMap[$pegawai->status_ptkp ?? 'TK/0'] ?? 54000000;
        $pkp = max(0, floor(($penghasilanNeto - $ptkp) / 1000) * 1000);

        $pph = 0;
        $sisa = $pkp;
        $lapisan = [
            [50000000, 0.05],
            [200000000, 0.15],
            [250000000, 0.25],
            [PHP_INT_MAX, 0.30],
        ];
        foreach ($lapisan as [$batas, $tarif]) {
            if ($sisa <= 0) break;
            $kena = min($sisa, $batas);
            $pph += $kena * $tarif;
            $sisa -= $kena;
        }

        $rekap = [
            'penghasilan_kotor' => $penghasilanKotor,
            'penghasilan_neto' => $penghasilanNeto,
            'ptkp' => $ptkp,
            'pkp' => $pkp,
            'pph_aktual' => $pph,
            'pph_terpotong' => $totalPotonganPPH,
            'selisih' => $totalPotonganPPH - $pph,
        ];

        $tahun = $gajis->first()?->bulan ? substr($gajis->first()->bulan, 0, 4) : '—';
        $totalGajiBersih = $totalGaji + $totalUangMakan + $totalTunjangan - $totalPotongan;
        $pdf = Pdf::loadView('gaji.slip_total', compact(
            'pegawai', 'gajis', 'rekap', 'totalGaji', 'totalUangMakan',
            'totalTunjangan', 'totalPotongan', 'totalPotonganPPH', 'totalGajiBersih', 'tahun'
        ))->setPaper('a4', 'landscape');

        return $pdf->stream('Slip-Total-Gaji-' . $pegawai->nama . '.pdf');
    }

    public function rekonsiliasiTahunan(Request $request)
{
    $tahun = $request->get('tahun', date('Y'));
    $pegawais = Pegawai::with(['gajis' => function ($query) use ($tahun) {
        $query->where('bulan', 'like', "$tahun%");
    }, 'gajis.tunjangan', 'gajis.potongan'])->get();

    $data = [];

    foreach ($pegawais as $pegawai) {
        $totalGajiPokok = $pegawai->gajis->sum('gaji_pokok');
        $totalUangMakan = $pegawai->gajis->sum('total_uang_makan');
        $totalTunjangan = $pegawai->gajis->flatMap->tunjangan->sum('jumlah');
        $totalPotongan = $pegawai->gajis->sum('total_potongan');
        $totalPPH21 = $pegawai->gajis->sum('pot_pph21');
        $totalGajiBersih = $totalGajiPokok + $totalUangMakan + $totalTunjangan - $totalPotongan;

        $penghasilanKotor = $totalGajiPokok + $totalUangMakan + $totalTunjangan;
        $biayaJabatan = min($penghasilanKotor * 0.05, 6000000);
        $penghasilanNeto = $penghasilanKotor - $biayaJabatan;

        $ptkpMap = [
    'TK/0' => 54000000,
    'TK/1' => 58500000,
    'TK/2' => 63000000,
    'TK/3' => 67500000,
    'K/0'  => 58500000,
    'K/1'  => 63000000,
    'K/2'  => 67500000,
    'K/3'  => 72000000,
        ];
        $ptkp = $ptkpMap[$pegawai->status_ptkp ?? 'TK/0'] ?? 54000000;
        $pkp = max(0, floor(($penghasilanNeto - $ptkp) / 1000) * 1000);

        $pph = 0;
        $sisa = $pkp;
        $lapisan = [
            [50000000, 0.05],
            [200000000, 0.15],
            [250000000, 0.25],
            [PHP_INT_MAX, 0.30],
        ];
        foreach ($lapisan as [$batas, $tarif]) {
            if ($sisa <= 0) break;
            $kena = min($sisa, $batas);
            $pph += $kena * $tarif;
            $sisa -= $kena;
        }

        $data[] = [
            'pegawai' => $pegawai,
            'totalGajiPokok' => $totalGajiPokok,
            'totalUangMakan' => $totalUangMakan,
            'totalTunjangan' => $totalTunjangan,
            'totalPotongan' => $totalPotongan,
            'totalPPH21' => $totalPPH21,
            'totalGajiBersih' => $totalGajiBersih,
            'penghasilan_kotor' => $penghasilanKotor,
            'penghasilan_neto' => $penghasilanNeto,
            'ptkp' => $ptkp,
            'pkp' => $pkp,
            'pph_aktual' => $pph,
            'selisih' => $totalPPH21 - $pph,
        ];
    }

    return view('gaji.rekonsiliasi', compact('data', 'tahun'));
}


    public function export()
    {
        return Excel::download(new GajiExport, 'data_gaji.xlsx');
    }

    public function importForm()
    {
        return view('gaji.import');
    }

    public function import(Request $request)
    {
        $request->validate([
            'file' => 'required|mimes:xls,xlsx',
        ]);

        $data = Excel::toArray([], $request->file('file'));

        foreach ($data[0] as $index => $row) {
            if ($index === 0 || count($row) < 6) continue;

            $bulan = $row[0];
            $nama_pegawai = $row[1];
            $jumlah_hadir = (int)$row[2];
            $uang_makan = (int)$row[3];
            $potongan = (int)$row[5];

            $pegawai = Pegawai::with('jabatan')->where('nama', $nama_pegawai)->first();
            if (!$pegawai) continue;

            $gaji_pokok = $pegawai->jabatan->gaji_pokok ?? 0;
            $total_gaji = $gaji_pokok + $uang_makan - $potongan;

            Gaji::updateOrCreate(
                ['pegawai_id' => $pegawai->id, 'bulan' => $bulan],
                [
                    'jumlah_hadir' => $jumlah_hadir,
                    'gaji_pokok' => $gaji_pokok,
                    'total_uang_makan' => $uang_makan,
                    'jumlah_pendapatan' => $gaji_pokok + $uang_makan,
                    'total_potongan' => $potongan,
                    'total_gaji' => $total_gaji,
                ]
            );

        }

        return redirect()->route('gaji.index')->with('success', 'Data gaji berhasil diimport.');
    }
}
