<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Pegawai;
use App\Models\UangMakan;
use App\Models\Gaji;
use App\Models\Presensi;
use App\Helpers\DateHelper;
use Carbon\Carbon;
use Barryvdh\DomPDF\Facade\Pdf;
use App\Http\Controllers\GajiController;

class UangMakanController extends Controller
{
    public function index()
    {
        $uangmakan = UangMakan::with('pegawai')->orderBy('bulan', 'desc')->get();
        return view('gaji.uangmakan.index', compact('uangmakan'));
    }

    public function create(Request $request)
    {
        $pegawai = Pegawai::all();
        $selectedPegawai = $request->get('pegawai_id');
        $selectedBulan = $request->get('bulan');
        $jumlahHadir = null;
        $maksHariKerja = null;

        if ($selectedPegawai && $selectedBulan) {
            $jumlahHadir = Presensi::where('pegawai_id', $selectedPegawai)
                ->whereMonth('tanggal', Carbon::parse($selectedBulan)->month)
                ->whereYear('tanggal', Carbon::parse($selectedBulan)->year)
                ->where('status', 'Hadir')
                ->count();

            $maksHariKerja = count(DateHelper::getTanggalHariKerja($selectedBulan));
        }

        return view('gaji.uangmakan.create', compact('pegawai', 'selectedPegawai', 'selectedBulan', 'jumlahHadir', 'maksHariKerja'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'pegawai_id'   => 'required|exists:pegawais,id',
            'bulan'        => 'required|date_format:Y-m',
            'jumlah_hari'  => 'required|integer|min:0',
            'tarif'        => 'required|integer|min:0',
        ]);

        $hariKerja = DateHelper::getTanggalHariKerja($request->bulan);
        $maxHari = count($hariKerja);

        if ($request->jumlah_hari > $maxHari) {
            return back()->withErrors([
                'jumlah_hari' => 'Jumlah hari hadir tidak boleh melebihi ' . $maxHari . ' hari kerja pada bulan tersebut.'
            ])->withInput();
        }

        $cek = UangMakan::where('pegawai_id', $request->pegawai_id)
                        ->where('bulan', $request->bulan)
                        ->first();

        if ($cek) {
            return back()->withErrors([
                'bulan' => 'Data uang makan untuk pegawai ini pada bulan tersebut sudah ada.'
            ])->withInput();
        }

        $jumlah = $request->jumlah_hari * $request->tarif;

        UangMakan::create([
            'pegawai_id'   => $request->pegawai_id,
            'bulan'        => $request->bulan,
            'jumlah_hari'  => $request->jumlah_hari,
            'tarif'        => $request->tarif,
            'jumlah'       => $jumlah,
        ]);

        $gaji = Gaji::where('pegawai_id', $request->pegawai_id)
            ->where('bulan', $request->bulan)
            ->first();

        if ($gaji) {
            app(GajiController::class)->hitungUlangPph($gaji);
        }

        return redirect()->route('uangmakan.laporan', ['periode' => $request->bulan])
                         ->with('success', 'Data uang makan berhasil disimpan.');
    }

    public function edit($id)
    {
        $uangmakan = UangMakan::findOrFail($id);
        $pegawai = Pegawai::all();
        return view('gaji.uangmakan.edit', compact('uangmakan', 'pegawai'));
    }

    public function update(Request $request, $id)
    {
        $request->validate([
            'pegawai_id'   => 'required|exists:pegawais,id',
            'bulan'        => 'required|date_format:Y-m',
            'jumlah_hari'  => 'required|integer|min:0',
            'tarif'        => 'required|integer|min:0',
        ]);

        $hariKerja = DateHelper::getTanggalHariKerja($request->bulan);
        $maxHari = count($hariKerja);

        if ($request->jumlah_hari > $maxHari) {
            return back()->withErrors([
                'jumlah_hari' => 'Jumlah hari hadir tidak boleh melebihi ' . $maxHari . ' hari kerja pada bulan tersebut.'
            ])->withInput();
        }

        $uangmakan = UangMakan::findOrFail($id);
        $jumlah = $request->jumlah_hari * $request->tarif;

        $uangmakan->update([
            'pegawai_id'   => $request->pegawai_id,
            'bulan'        => $request->bulan,
            'jumlah_hari'  => $request->jumlah_hari,
            'tarif'        => $request->tarif,
            'jumlah'       => $jumlah,
        ]);

        $gaji = Gaji::where('pegawai_id', $request->pegawai_id)
            ->where('bulan', $request->bulan)
            ->first();

        if ($gaji) {
            app(GajiController::class)->hitungUlangPph($gaji);
        }

        return redirect()->route('uangmakan.laporan', ['periode' => $request->bulan])
                         ->with('success', 'Data uang makan berhasil diperbarui.');
    }

    public function laporan(Request $request)
    {
        $semuaPeriode = UangMakan::select('bulan')->distinct()->orderBy('bulan', 'desc')->pluck('bulan');
        $periode = $request->get('periode', $semuaPeriode->first());

        $uangmakan = collect();
        if ($periode) {
            $uangmakan = UangMakan::with('pegawai')->where('bulan', $periode)->get();
        }

        return view('gaji.uangmakan.laporan', compact('uangmakan', 'periode', 'semuaPeriode'));
    }

    public function filter(Request $request)
    {
        $request->validate([
            'periode' => 'required|date_format:Y-m',
        ]);

        return redirect()->route('uangmakan.laporan', ['periode' => $request->periode]);
    }

    public function cetakPdf(Request $request)
    {
        $request->validate([
            'bulan' => 'required',
            'tahun' => 'required',
        ]);

        $bulan = $request->bulan;
        $tahun = $request->tahun;
        $periode = $tahun . '-' . str_pad($bulan, 2, '0', STR_PAD_LEFT);

        $uangmakan = UangMakan::with('pegawai')->where('bulan', $periode)->get();

        $pdf = Pdf::loadView('gaji.uangmakan.pdf', compact('uangmakan', 'periode', 'bulan', 'tahun'));
        return $pdf->download('laporan-uang-makan-' . $periode . '.pdf');
    }
}
