<?php

namespace App\Http\Controllers;

use App\Models\Pegawai;
use App\Models\Jabatan;
use App\Models\JobHistory;
use Illuminate\Http\Request;

class PegawaiController extends Controller
{
    public function index(Request $request)
    {
        $keyword = $request->input('search');

        $pegawais = Pegawai::when($keyword, function ($query, $keyword) {
            $query->where('nama', 'like', "%$keyword%")
                  ->orWhere('nip', 'like', "%$keyword%");
        })
        ->orderBy('nama')
        ->paginate(10);

        return view('pegawai.index', compact('pegawais', 'keyword'));
    }

    public function create()
    {
        $jabatans = Jabatan::all();
        return view('pegawai.create', compact('jabatans'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'nama' => 'required',
            'nip' => 'required|unique:pegawais',
            'jabatan_id' => 'required|exists:jabatans,id',
            'divisi' => 'required',
            'status_karyawan' => 'required|in:Tetap,Kontrak',
            'tanggal_masuk' => 'nullable|date',
            'pendidikan_terakhir' => 'nullable|string|max:100',
            'uang_makan_per_hari' => 'nullable|numeric',
            'gaji_pokok' => 'nullable|numeric',
        ]);

        Pegawai::create($request->all());

        return redirect()->route('pegawai.index')->with('success', 'Pegawai berhasil ditambahkan.');
    }

    public function edit(Pegawai $pegawai)
    {
        $jabatans = Jabatan::all();
        return view('pegawai.edit', compact('pegawai', 'jabatans'));
    }

    public function update(Request $request, Pegawai $pegawai)
    {
        $request->validate([
            'nama' => 'required',
            'nip' => 'required|unique:pegawais,nip,' . $pegawai->id,
            'jabatan_id' => 'required|exists:jabatans,id',
            'divisi' => 'required',
            'status_karyawan' => 'required|in:Tetap,Kontrak',
            'tanggal_masuk' => 'nullable|date',
            'pendidikan_terakhir' => 'nullable|string|max:100',
            'uang_makan_per_hari' => 'nullable|numeric',
        ]);

        $jabatanLama = $pegawai->jabatan?->nama;
        $gajiLama = $pegawai->jabatan?->gaji_pokok;

        $jabatanBaruModel = Jabatan::find($request->jabatan_id);
        $jabatanBaru = $jabatanBaruModel?->nama;
        $gajiBaru = $jabatanBaruModel?->gaji_pokok;

        $pegawai->update($request->all());

        if ($jabatanLama !== $jabatanBaru || $gajiLama != $gajiBaru) {
            JobHistory::create([
                'pegawai_id' => $pegawai->id,
                'jabatan_lama' => $jabatanLama,
                'gaji_lama' => $gajiLama,
                'jabatan_baru' => $jabatanBaru,
                'gaji_baru' => $gajiBaru,
                'tanggal_berlaku' => now()->toDateString(),
                'keterangan' => 'Perubahan melalui edit data pegawai',
            ]);
        }

        return redirect()->route('pegawai.index')->with('success', 'Data pegawai berhasil diperbarui.');
    }

    public function destroy(Pegawai $pegawai)
    {
        $pegawai->delete();
        return redirect()->route('pegawai.index')->with('success', 'Pegawai berhasil dihapus.');
    }

    public function show(Pegawai $pegawai)
    {
        return view('pegawai.show', compact('pegawai'));
    }

    public function bulkDelete(Request $request)
    {
        $request->validate([
            'ids' => 'required|array',
            'ids.*' => 'exists:pegawais,id',
        ]);

        Pegawai::whereIn('id', $request->ids)->delete();

        return redirect()->route('pegawai.index')->with('success', 'Beberapa pegawai berhasil dihapus.');
    }
}
